function validateAccessCode(code) {
  const validCodes = (process.env.ACCESS_CODES || '').split(',').map(c => c.trim().toLowerCase());
  if (validCodes.length === 0 || (validCodes.length === 1 && validCodes[0] === '')) {
    return true;
  }
  return validCodes.includes((code || '').trim().toLowerCase());
}

export default async function handler(req, res) {
  if (req.method !== 'POST') {
    return res.status(405).json({ error: 'Method not allowed' });
  }

  const apiKey = process.env.ASSEMBLYAI_API_KEY;
  if (!apiKey) {
    console.error('ASSEMBLYAI_API_KEY not configured');
    return res.status(500).json({ error: 'Service not configured.' });
  }

  // Validate access code
  const accessCode = req.headers['x-access-code'];
  if (!validateAccessCode(accessCode)) {
    return res.status(403).json({ error: 'Invalid access code.' });
  }

  const { audio_url } = req.body;
  
  // Validate audio_url format (should be AssemblyAI upload URL)
  if (!audio_url || typeof audio_url !== 'string') {
    return res.status(400).json({ error: 'Invalid request.' });
  }
  
  if (!audio_url.startsWith('https://cdn.assemblyai.com/')) {
    return res.status(400).json({ error: 'Invalid audio source.' });
  }

  try {
    const response = await fetch('https://api.assemblyai.com/v2/transcript', {
      method: 'POST',
      headers: {
        'Authorization': apiKey,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        audio_url: audio_url,
        speaker_labels: true,
      }),
    });

    const data = await response.json();
    
    if (!response.ok) {
      console.error('AssemblyAI transcribe error:', data);
      return res.status(response.status).json({ error: 'Transcription request failed.' });
    }

    console.log(`Transcription started: ${data.id}, time: ${new Date().toISOString()}`);
    return res.status(200).json(data);
  } catch (error) {
    console.error('Transcribe error:', error.message);
    return res.status(500).json({ error: 'Transcription request failed.' });
  }
}
